// Photoshop: Convert Images to Black & White with Optional Contrast Boost
// Processes all images in a selected folder and saves them as JPEGs in an output folder

// ------------------------------------------------------
// MAIN FUNCTION
// ------------------------------------------------------
function main() {

    // --- Step 1: Ask user to select source folder ---
    alert("Please select the folder containing the images you want to convert to black & white.");
    var sourceFolder = Folder.selectDialog("Select the folder containing your images:");
    if (!sourceFolder) return;

    // --- Step 2: Ask user to select or create output folder ---
    alert("Now, please select (or create) the folder where the converted images will be saved.");
    var outputFolder = Folder.selectDialog("Select the folder to save converted images:");
    if (!outputFolder) return;

    // --- Step 3: Ask if user wants contrast boost ---
    var addContrast = confirm("Do you want to add a +20 contrast boost to the images?");

    // Get list of supported images
    var files = sourceFolder.getFiles(/\.(jpg|jpeg|tif|tiff|png|psd)$/i);

    if (files.length === 0) {
        alert("No supported image files found in the selected folder.");
        return;
    }

    // Process each image
    for (var i = 0; i < files.length; i++) {
        try {
            var file = files[i];
            var doc = app.open(file);

            // Convert to Black & White
            convertToBlackAndWhite(doc);

            // Apply contrast boost if requested
            if (addContrast) {
                adjustContrast(doc, 20);
            }

            // Save as JPEG
            saveAsJPG(doc, outputFolder);

            doc.close(SaveOptions.DONOTSAVECHANGES);

        } catch (e) {
            alert("Error processing file: " + file.name + "\n" + e.message);
        }
    }

    alert("All images have been converted to black & white and saved in:\n" + outputFolder.fsName);
}

// ------------------------------------------------------
// CONVERT TO BLACK & WHITE FUNCTION
// ------------------------------------------------------
function convertToBlackAndWhite(doc) {
    // Convert to Grayscale mode
    doc.changeMode(ChangeMode.GRAYSCALE);
    
    // Convert back to RGB so we can save as JPEG with color profile
    doc.changeMode(ChangeMode.RGB);
}

// ------------------------------------------------------
// ADJUST CONTRAST FUNCTION
// ------------------------------------------------------
function adjustContrast(doc, contrastAmount) {
    // Use Brightness/Contrast adjustment
    var idBrghtnss = charIDToTypeID("BrgC");
    var desc = new ActionDescriptor();
    var idBrightness = charIDToTypeID("Brgh");
    var idContrast = charIDToTypeID("Cntr");
    var iduseLegacy = stringIDToTypeID("useLegacy");
    
    desc.putInteger(idBrightness, 0);
    desc.putInteger(idContrast, contrastAmount);
    desc.putBoolean(iduseLegacy, false);
    
    executeAction(idBrghtnss, desc, DialogModes.NO);
}

// ------------------------------------------------------
// SAVE AS JPEG FUNCTION
// ------------------------------------------------------
function saveAsJPG(doc, outputFolder) {
    var baseName = doc.name.replace(/\.[^\.]+$/, "");
    var outFile = new File(outputFolder + "/" + baseName + "_bw.jpg");

    var opts = new JPEGSaveOptions();
    opts.quality = 12;
    opts.embedColorProfile = true;

    doc.saveAs(outFile, opts, true, Extension.LOWERCASE);
}

// ------------------------------------------------------
// RUN THE SCRIPT
// ------------------------------------------------------
main();